/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.util;

import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.jspecify.annotations.Nullable;
import org.springframework.util.Assert;

public final class ConcurrentLruCache<K, V> {
    private final int capacity;
    private final AtomicInteger currentSize = new AtomicInteger();
    private final ConcurrentMap<K, Node<K, V>> cache;
    private final Function<K, V> generator;
    private final ReadOperations<K, V> readOperations;
    private final WriteOperations writeOperations;
    private final Lock evictionLock = new ReentrantLock();
    private final EvictionQueue<K, V> evictionQueue = new EvictionQueue();
    private final AtomicReference<DrainStatus> drainStatus = new AtomicReference<DrainStatus>(DrainStatus.IDLE);

    public ConcurrentLruCache(int capacity, Function<K, V> generator) {
        this(capacity, generator, 16);
    }

    private ConcurrentLruCache(int capacity, Function<K, V> generator, int concurrencyLevel) {
        Assert.isTrue(capacity >= 0, "Capacity must be >= 0");
        this.capacity = capacity;
        this.cache = new ConcurrentHashMap<K, Node<K, V>>(16, 0.75f, concurrencyLevel);
        this.generator = generator;
        this.readOperations = new ReadOperations<K, V>(this.evictionQueue);
        this.writeOperations = new WriteOperations();
    }

    public V get(K key) {
        if (this.capacity == 0) {
            return this.generator.apply(key);
        }
        Node node = (Node)this.cache.get(key);
        if (node == null) {
            V value = this.generator.apply(key);
            this.put(key, value);
            return value;
        }
        this.processRead(node);
        return node.getValue();
    }

    private void put(K key, V value) {
        Assert.notNull(key, "key must not be null");
        Assert.notNull(value, "value must not be null");
        CacheEntry<V> cacheEntry = new CacheEntry<V>(value, CacheEntryState.ACTIVE);
        Node<K, V> node = new Node<K, V>(key, cacheEntry);
        Node<K, V> prior = this.cache.putIfAbsent(node.key, node);
        if (prior == null) {
            this.processWrite(new AddTask(node));
        } else {
            this.processRead(prior);
        }
    }

    private void processRead(Node<K, V> node) {
        boolean drainRequested = this.readOperations.recordRead(node);
        DrainStatus status = this.drainStatus.get();
        if (status.shouldDrainBuffers(drainRequested)) {
            this.drainOperations();
        }
    }

    private void processWrite(Runnable task) {
        this.writeOperations.add(task);
        this.drainStatus.lazySet(DrainStatus.REQUIRED);
        this.drainOperations();
    }

    private void drainOperations() {
        if (this.evictionLock.tryLock()) {
            try {
                this.drainStatus.lazySet(DrainStatus.PROCESSING);
                this.readOperations.drain();
                this.writeOperations.drain();
            }
            finally {
                this.drainStatus.compareAndSet(DrainStatus.PROCESSING, DrainStatus.IDLE);
                this.evictionLock.unlock();
            }
        }
    }

    public int capacity() {
        return this.capacity;
    }

    @Deprecated(since="6.0")
    public int sizeLimit() {
        return this.capacity;
    }

    public int size() {
        return this.cache.size();
    }

    public void clear() {
        this.evictionLock.lock();
        try {
            Node<K, V> node;
            while ((node = this.evictionQueue.poll()) != null) {
                this.cache.remove(node.key, node);
                this.markAsRemoved(node);
            }
            this.readOperations.clear();
            this.writeOperations.drainAll();
        }
        finally {
            this.evictionLock.unlock();
        }
    }

    private void markAsRemoved(Node<K, V> node) {
        CacheEntry removed;
        CacheEntry current;
        while (!node.compareAndSet(current = (CacheEntry)node.get(), removed = new CacheEntry(current.value, CacheEntryState.REMOVED))) {
        }
        this.currentSize.lazySet(this.currentSize.get() - 1);
    }

    public boolean contains(K key) {
        return this.cache.containsKey(key);
    }

    public boolean remove(K key) {
        Node node = (Node)this.cache.remove(key);
        if (node == null) {
            return false;
        }
        this.markForRemoval(node);
        this.processWrite(new RemovalTask(node));
        return true;
    }

    private void markForRemoval(Node<K, V> node) {
        CacheEntry pendingRemoval;
        CacheEntry current;
        do {
            if ((current = (CacheEntry)node.get()).isActive()) continue;
            return;
        } while (!node.compareAndSet(current, pendingRemoval = new CacheEntry(current.value, CacheEntryState.PENDING_REMOVAL)));
    }

    private static final class EvictionQueue<K, V> {
        @Nullable Node<K, V> first;
        @Nullable Node<K, V> last;

        private EvictionQueue() {
        }

        @Nullable Node<K, V> poll() {
            if (this.first == null) {
                return null;
            }
            Node f = this.first;
            Node next = f.getNext();
            f.setNext(null);
            this.first = next;
            if (next == null) {
                this.last = null;
            } else {
                next.setPrevious(null);
            }
            return f;
        }

        void add(Node<K, V> e) {
            if (this.contains(e)) {
                return;
            }
            this.linkLast(e);
        }

        private boolean contains(Node<K, V> e) {
            return e.getPrevious() != null || e.getNext() != null || e == this.first;
        }

        private void linkLast(Node<K, V> e) {
            Node<K, V> l = this.last;
            this.last = e;
            if (l == null) {
                this.first = e;
            } else {
                l.setNext(e);
                e.setPrevious(l);
            }
        }

        private void unlink(Node<K, V> e) {
            Node<K, V> prev = e.getPrevious();
            Node<K, V> next = e.getNext();
            if (prev == null) {
                this.first = next;
            } else {
                prev.setNext(next);
                e.setPrevious(null);
            }
            if (next == null) {
                this.last = prev;
            } else {
                next.setPrevious(prev);
                e.setNext(null);
            }
        }

        void moveToBack(Node<K, V> e) {
            if (this.contains(e) && e != this.last) {
                this.unlink(e);
                this.linkLast(e);
            }
        }

        void remove(Node<K, V> e) {
            if (this.contains(e)) {
                this.unlink(e);
            }
        }
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    private static enum DrainStatus {
        IDLE{

            @Override
            boolean shouldDrainBuffers(boolean delayable) {
                return !delayable;
            }
        }
        ,
        REQUIRED{

            @Override
            boolean shouldDrainBuffers(boolean delayable) {
                return true;
            }
        }
        ,
        PROCESSING{

            @Override
            boolean shouldDrainBuffers(boolean delayable) {
                return false;
            }
        };


        abstract boolean shouldDrainBuffers(boolean var1);
    }

    private static final class ReadOperations<K, V> {
        private static final int BUFFER_COUNT = ReadOperations.detectNumberOfBuffers();
        private static final int BUFFERS_MASK = BUFFER_COUNT - 1;
        private static final int MAX_PENDING_OPERATIONS = 32;
        private static final int MAX_DRAIN_COUNT = 64;
        private static final int BUFFER_SIZE = 128;
        private static final int BUFFER_INDEX_MASK = 127;
        private final AtomicLongArray recordedCount = new AtomicLongArray(BUFFER_COUNT);
        private final long[] readCount = new long[BUFFER_COUNT];
        private final AtomicLongArray processedCount = new AtomicLongArray(BUFFER_COUNT);
        private final AtomicReferenceArray<Node<K, V>>[] buffers = new AtomicReferenceArray[BUFFER_COUNT];
        private final EvictionQueue<K, V> evictionQueue;

        private static int detectNumberOfBuffers() {
            int availableProcessors = Runtime.getRuntime().availableProcessors();
            int nextPowerOfTwo = 1 << 32 - Integer.numberOfLeadingZeros(availableProcessors - 1);
            return Math.min(4, nextPowerOfTwo);
        }

        ReadOperations(EvictionQueue<K, V> evictionQueue) {
            this.evictionQueue = evictionQueue;
            for (int i2 = 0; i2 < BUFFER_COUNT; ++i2) {
                this.buffers[i2] = new AtomicReferenceArray(128);
            }
        }

        private static int getBufferIndex() {
            return (int)Thread.currentThread().getId() & BUFFERS_MASK;
        }

        boolean recordRead(Node<K, V> node) {
            int bufferIndex = ReadOperations.getBufferIndex();
            long writeCount = this.recordedCount.get(bufferIndex);
            this.recordedCount.lazySet(bufferIndex, writeCount + 1L);
            int index = (int)(writeCount & 0x7FL);
            this.buffers[bufferIndex].lazySet(index, node);
            long pending = writeCount - this.processedCount.get(bufferIndex);
            return pending < 32L;
        }

        void drain() {
            int start = (int)Thread.currentThread().getId();
            int end = start + BUFFER_COUNT;
            for (int i2 = start; i2 < end; ++i2) {
                this.drainReadBuffer(i2 & BUFFERS_MASK);
            }
        }

        void clear() {
            for (int i2 = 0; i2 < BUFFER_COUNT; ++i2) {
                AtomicReferenceArray<Node<K, V>> buffer = this.buffers[i2];
                for (int j = 0; j < 128; ++j) {
                    buffer.lazySet(j, null);
                }
            }
        }

        private void drainReadBuffer(int bufferIndex) {
            int index;
            AtomicReferenceArray<Node<K, V>> buffer;
            Node<K, V> node;
            long writeCount = this.recordedCount.get(bufferIndex);
            for (int i2 = 0; i2 < 64 && (node = (buffer = this.buffers[bufferIndex]).get(index = (int)(this.readCount[bufferIndex] & 0x7FL))) != null; ++i2) {
                buffer.lazySet(index, null);
                this.evictionQueue.moveToBack(node);
                int n = bufferIndex;
                this.readCount[n] = this.readCount[n] + 1L;
            }
            this.processedCount.lazySet(bufferIndex, writeCount);
        }
    }

    private static final class WriteOperations {
        private static final int DRAIN_THRESHOLD = 16;
        private final Queue<Runnable> operations = new ConcurrentLinkedQueue<Runnable>();

        private WriteOperations() {
        }

        public void add(Runnable task) {
            this.operations.add(task);
        }

        public void drain() {
            Runnable task;
            for (int i2 = 0; i2 < 16 && (task = this.operations.poll()) != null; ++i2) {
                task.run();
            }
        }

        public void drainAll() {
            Runnable task;
            while ((task = this.operations.poll()) != null) {
                task.run();
            }
        }
    }

    private static final class Node<K, V>
    extends AtomicReference<CacheEntry<V>> {
        final K key;
        @Nullable Node<K, V> prev;
        @Nullable Node<K, V> next;

        Node(K key, CacheEntry<V> cacheEntry) {
            super(cacheEntry);
            this.key = key;
        }

        public @Nullable Node<K, V> getPrevious() {
            return this.prev;
        }

        public void setPrevious(@Nullable Node<K, V> prev) {
            this.prev = prev;
        }

        public @Nullable Node<K, V> getNext() {
            return this.next;
        }

        public void setNext(@Nullable Node<K, V> next) {
            this.next = next;
        }

        V getValue() {
            return ((CacheEntry)this.get()).value;
        }
    }

    private record CacheEntry<V>(V value, CacheEntryState state) {
        boolean isActive() {
            return this.state == CacheEntryState.ACTIVE;
        }
    }

    private static enum CacheEntryState {
        ACTIVE,
        PENDING_REMOVAL,
        REMOVED;

    }

    private final class AddTask
    implements Runnable {
        final Node<K, V> node;

        AddTask(Node<K, V> node) {
            this.node = node;
        }

        @Override
        public void run() {
            ConcurrentLruCache.this.currentSize.lazySet(ConcurrentLruCache.this.currentSize.get() + 1);
            if (((CacheEntry)this.node.get()).isActive()) {
                ConcurrentLruCache.this.evictionQueue.add(this.node);
                this.evictEntries();
            }
        }

        private void evictEntries() {
            while (ConcurrentLruCache.this.currentSize.get() > ConcurrentLruCache.this.capacity) {
                Node node = ConcurrentLruCache.this.evictionQueue.poll();
                if (node == null) {
                    return;
                }
                ConcurrentLruCache.this.cache.remove(node.key, node);
                ConcurrentLruCache.this.markAsRemoved(node);
            }
        }
    }

    private final class RemovalTask
    implements Runnable {
        final Node<K, V> node;

        RemovalTask(Node<K, V> node) {
            this.node = node;
        }

        @Override
        public void run() {
            ConcurrentLruCache.this.evictionQueue.remove(this.node);
            ConcurrentLruCache.this.markAsRemoved(this.node);
        }
    }
}

